//
//  NewViewController.swift
//  UaePassDemo
//
//  Created by Mohammed Gomaa on 1/2/19.
//  Copyright © 2019 Mohammed Gomaa. All rights reserved.
//

import UIKit
import UAEPassClient

@objc public class ViewController: UIViewController {
    @objc public var uaePassAccessToken: String!
    var env: UAEPASSEnvirnonment!
    var textFields: [UITextField] = []

    @IBOutlet var loginButton: UIButton!
    
    
    @IBOutlet var clientIDTF: UITextField!
    @IBOutlet var clientSecretTF: UITextField!
    @IBOutlet var redirectURLTF: UITextField!
    @IBOutlet var scopeTF: UITextField!
    @IBOutlet var successURLTF: UITextField!
    @IBOutlet var failureURLTF: UITextField!
    @IBOutlet var signinScopeTF: UITextField!
    
    @IBOutlet var languageSegment: UISegmentedControl!
    @IBOutlet var envSegment: UISegmentedControl!

    
    var activeTextField: UITextField?

    public override func viewDidLoad() {
        UAEPASSRouter.shared.sdkLang = .english // .arabic for Arabic
        env = .staging // Default environment which SPs should ONLY use it for testing (QA and DEV environments not suppose to be working for SPs.)
        languageSegment.selectedSegmentIndex = 0 
        languageSegment.addTarget(self, action: #selector(langValueChanged(_:)), for: .valueChanged)

        envSegment.selectedSegmentIndex = 1 
        envSegment.addTarget(self, action: #selector(envValueChanged(_:)), for: .valueChanged)

        
        // Add text fields to the array
        textFields = [clientIDTF, clientSecretTF, redirectURLTF, scopeTF, successURLTF, failureURLTF, signinScopeTF]
        
        // Set the text fields' delegate
        for textField in textFields {
            textField.delegate = self
        }

        let tapGesture = UITapGestureRecognizer(target: self, action: #selector(dismissKeyboard))
        view.addGestureRecognizer(tapGesture)
        
        NotificationCenter.default.addObserver(self, selector: #selector(keyboardWillShow), name: UIResponder.keyboardWillShowNotification, object: nil)
        NotificationCenter.default.addObserver(self, selector: #selector(keyboardWillHide), name: UIResponder.keyboardWillHideNotification, object: nil)
        
    }
    
    
    deinit {
        NotificationCenter.default.removeObserver(self, name: UIResponder.keyboardWillShowNotification, object: nil)
        NotificationCenter.default.removeObserver(self, name: UIResponder.keyboardWillHideNotification, object: nil)
    }
    
    @objc func keyboardWillShow(notification: NSNotification) {
        if let keyboardFrame = notification.userInfo?[UIResponder.keyboardFrameEndUserInfoKey] as? NSValue {
            let keyboardHeight = keyboardFrame.cgRectValue.height
            // Adjust the view's bottom constraint
            if activeTextField == clientIDTF ||
                activeTextField == clientSecretTF ||
                activeTextField == redirectURLTF ||
                activeTextField == scopeTF {
            } else {
                adjustViewForKeyboard(show: true, keyboardHeight: keyboardHeight)   
            }
        }
    }

    @objc func keyboardWillHide(notification: NSNotification) {
        // Adjust the view's bottom constraint
        adjustViewForKeyboard(show: false, keyboardHeight: 0)
    }

    func adjustViewForKeyboard(show: Bool, keyboardHeight: CGFloat) {
        // Adjust the view's bottom constraint or frame based on the keyboard height
        if show {
            // Move the view up by keyboard height
            view.frame.origin.y = -keyboardHeight
        } else {
            // Reset the view's position
            view.frame.origin.y = 0
        }
    }
    
    @objc func dismissKeyboard() {
        view.endEditing(true)
    }
    
    // MARK: - Select environment -
    /// sample method just to be able to select environment type.
    @IBAction func actionLoginWithUaePass(_ sender: Any) {
        UAEPASSRouter.shared.spConfig = SPConfig(redirectUriLogin: redirectURLTF.text ?? "",
                                                 scope: scopeTF.text ?? "",
                                                 state: "UzkLPOzcsHs4SL9cNZ7bFATd",  //Randomly Generated Code 24 alpha numeric.
                                                 successSchemeURL: successURLTF.text ?? "",
                                                 failSchemeURL: failureURLTF.text ?? "",
                                                 signingScope: signinScopeTF.text ?? "")
        UAEPASSRouter.shared.environmentConfig = UAEPassConfig(clientID: clientIDTF.text ?? "", clientSecret: clientSecretTF.text ?? "", env: env)
        self.login()
    }
    
    // MARK: - Getting -
    /// sample method just to be able to select environment type.
    ///
    
    func login(isFaceAuth: Bool? = false) {
        self.view.showBlurLoader()
        self.view.removeBluerLoader()
        if let webVC = UAEPassWebViewController.instantiate() as? UAEPassWebViewController {
            webVC.urlString = UAEPassConfiguration.getServiceUrlForType(serviceType: .loginURL)
            webVC.onUAEPassSuccessBlock = {(code: String?) -> Void in
                self.navigationController?.dismiss(animated: true)
                if let code = code {
                    self.getUaePassTokenForCode(code: code)
                }
            }
            webVC.onUAEPassFailureBlock = {(response: String?) -> Void in
                self.navigationController?.dismiss(animated: true)
            }
            webVC.reloadwithURL(url: webVC.urlString)
            self.navigationController?.present(webVC, animated: true)
        }
    }
    
    
    @objc func langValueChanged(_ sender: UISegmentedControl) {
        switch sender.selectedSegmentIndex {
        case 0:
            UAEPASSRouter.shared.sdkLang = .english
            print("English")
        case 1:
            UAEPASSRouter.shared.sdkLang = .arabic
            print("عربي")
        default:
            break
        }
    }
    
    
    @objc func envValueChanged(_ sender: UISegmentedControl) {
        switch sender.selectedSegmentIndex {
        case 0:
            env = .production
            print("production")
        case 1:
            env = .staging        
            print("staging")
        case 2:
            env = .qa        
            print("QA .. supposed to be for internal testing only.")
        case 3:
            env = .dev        
            print("For internal testing only..")

        default:
            break
        }
    }
    
    func getUaePassTokenForCode(code: String) {
        self.view.showBlurLoader()
        UAEPASSNetworkRequests.shared.getUAEPassToken(code: code, completion: { (uaePassToken) in
            self.view.removeBluerLoader()
            if let uaePassToken = uaePassToken, let accessToken = uaePassToken.accessToken {
                self.uaePassAccessToken = accessToken
                self.getUaePassProfileForToken(token: accessToken)
            } else {
                self.showErrorAlert(title: "Error", message: "Unable to get user token, Please try again.")
                return
            }
        }) { (error) in
            self.view.removeBluerLoader()
            self.showErrorAlert(title: "Error", message: error.value())
        }
    }
    
    func getUaePassProfileForToken(token: String) {
        UAEPASSNetworkRequests.shared.getUAEPassUserProfile(token: token, completion: { (userProfile) in
            if let userProfile = userProfile {
                self.showProfileDetails(userProfile: userProfile, userToken: token)
            } else {
                self.showErrorAlert(title: "Error", message: "Couldn't get user profile, Please try again later")
            }
        }) { (error) in
            self.showErrorAlert(title: "Error", message: error.value())
        }
    }
    
    func showProfileDetails(userProfile: UAEPassUserProfile, userToken: String) {
        let userProfileVC = self.storyboard?.instantiateViewController(withIdentifier: "UserProfileViewController") as? UserProfileViewController
        userProfileVC?.userProfile = userProfile
        userProfileVC?.userToken = userToken
        if let userProfileVC = userProfileVC {
            self.navigationController?.pushViewController(userProfileVC, animated: true)
        } else {
            self.showErrorAlert(title: "Error", message: "Can't find User Profile View, Please check your storyboard")
        }
    }
    
    func showErrorAlert(title: String, message: String) {
        let alertController = UIAlertController(title: "Error", message: message, preferredStyle: .alert)
        let action1 = UIAlertAction(title: "Ok", style: .default) { (action:UIAlertAction) in
        }
        alertController.addAction(action1)
        self.present(alertController, animated: true, completion: nil)
    }
}

extension UIView {
    func showBlurLoader(message: String? = "") {
        let blurLoader = BlurLoader(frame: frame, message: message)
        self.addSubview(blurLoader)
    }
    
    func removeBluerLoader() {
        if let blurLoader = subviews.first(where: { $0 is BlurLoader }) {
            blurLoader.removeFromSuperview()
        }
    }
}


class BlurLoader: UIView {
    
    var blurEffectView: UIVisualEffectView?
    
    init(frame: CGRect, message: String? = "") {
        let blurEffect = UIBlurEffect(style: .dark)
        let blurEffectView = UIVisualEffectView(effect: blurEffect)
        blurEffectView.frame = frame
        blurEffectView.autoresizingMask = [.flexibleWidth, .flexibleHeight]
        self.blurEffectView = blurEffectView
        super.init(frame: frame)
        addSubview(blurEffectView)
        addLoader()
    }
    
    override init(frame: CGRect) {
        let blurEffect = UIBlurEffect(style: .dark)
        let blurEffectView = UIVisualEffectView(effect: blurEffect)
        blurEffectView.frame = frame
        blurEffectView.autoresizingMask = [.flexibleWidth, .flexibleHeight]
        self.blurEffectView = blurEffectView
        super.init(frame: frame)
        addSubview(blurEffectView)
        addLoader()
    }
    
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    private func addLoader() {
        guard let blurEffectView = blurEffectView else { return }
        let activityIndicator = UIActivityIndicatorView(style: .large)
        activityIndicator.frame = CGRect(x: 0, y: 0, width: 50, height: 50)
        blurEffectView.contentView.addSubview(activityIndicator)
        activityIndicator.center = blurEffectView.contentView.center
        activityIndicator.startAnimating()
    }
}


extension ViewController: UITextFieldDelegate { 
    public func textFieldDidBeginEditing(_ textField: UITextField) {
        activeTextField = textField
    }

    public func textFieldDidEndEditing(_ textField: UITextField) {
        activeTextField = nil
    }
    
    public func textFieldShouldReturn(_ textField: UITextField) -> Bool {        
        if let index = textFields.firstIndex(of: textField), index < textFields.count - 1 {
            textFields[index + 1].becomeFirstResponder()
        } else {
            textField.resignFirstResponder()
        }
        return true
    }
}
